function [p, atom] = ga_atom(sig, t, atomfct, p0, nb_iter, p_up)
% Manifold Gradient Ascent convergence for any real atom of a dictionary
%
% INPUTS:
%  sig : orginal signal to fit
%  t : the time vector, evenly spaced
%  atomfct : function handle linked to the atom function. atomfct MUST
%  take as parameters (t, p0) in this order !
%  p0 : initial parameter for the convergence
%  nb_iter : number of iteration for the convergence
%  p_up : position of the parameter to update (default: 1:length(p0))
%
% OUTPUTS:
%  p : new parameter obtained from NG
%  atom : corresponding atom
%
% REQUIRES:
%  makecol 

sig = makecol(sig);
p = makecol(p0);


if (~exist('p_up'))
  nb_param = length(p0);
  p_up = 1:nb_param;
else
  nb_param = length(p_up);
end

T = t(2) - t(1);
lg = length(sig);

atom = makecol(atomfct(t, p));
sig_dot_atom = T*atom'*sig;
%fprintf('|<sig|atom>|: %f\n', abs(sig_dot_atom));

epsilon = 0.00001;
alpha = 0.01;

for iter = 1:nb_iter,

  %% Computing the gradient and the Hessian
  %% using a bunch of matrix computation
  %% For full information, see the paper 
  %% (Eqt (22)-(30))
  J = zeros(nb_param, 1);
  datom = zeros(lg, nb_param);
  
  for k = 1:nb_param,
    np = p;
    np(p_up(k)) = np(p_up(k)) + epsilon;
    natom = makecol(atomfct(t, np));
    datom(:,k) = (natom - atom)/epsilon;
  end
  
  %% The Gradient (forget the positive constant since normalization
  %% after by the norm of J)
  J = sig_dot_atom * (datom)'*sig;
  
  %% The metric
  G = (T*datom'*datom);
  
  %% Inverse of the Hessian is computed by pseudoinverse for
  %% stability
  Gpinv = inv(G);
  normJ = (J'*Gpinv*J)^.5;
  
  %% Optimization step
  dp_up = (Gpinv*J/normJ)*alpha;
  
  np(p_up) = p(p_up) + dp_up;
  
  natom = makecol(atomfct(t, np));
  new_sig_dot_atom = T*natom'*sig;
  good_update = abs(new_sig_dot_atom) > abs(sig_dot_atom);
  
  %% Line search of the best optimization step size
  %% If not truly an optimization of the scalar product, we divide
  %% the step size by 2 until to reach a true gain
% $$$   new_sig_dot_atom = 0;
% $$$   good_update = 0;  
% $$$ 
% $$$   good_update = 0;
% $$$   for s = 0,
% $$$     if (abs(new_sig_dot_atom) < abs(sig_dot_atom))
% $$$       np = p;
% $$$       np(p_up) = p(p_up) + dp_up/2^s;
% $$$       natom = makecol(atomfct(t, np));
% $$$       new_sig_dot_atom = T*natom'*sig;
% $$$     else
% $$$       good_update = 1;
% $$$       break
% $$$     end
% $$$   end
  
  %% checking if it is worth to continue
  if (good_update)
    p = np;
    sig_dot_atom = new_sig_dot_atom;
    atom = natom;
    %fprintf('|<sig|atom>|: %f\n', abs(sig_dot_atom));
  else
    %disp('Nothing to search anymore');
    break;
  end  
  %end
end
