function [atoms, last_resid] = mpdecomp(sig, nbiter, nbscales, phi)
% Matching pursuit decomposition of 1-D signal using discrete dico
% based on Mexican Hat
%
% See also : mex1d.m and mex1d_freq.m

% INPUTS: 
% * sig : the original signal
% * nbiter : the number of iterations
% * nbscales : the number of scales between 1 length(sig)/2
%
% OUTPUTS:
% * atoms : a nbiter x 4 matrix where column 1 gives the scale
% index, column 2 the scale value, column 3 the position and column
% 4 the scalar product of the best atom at each iteration.
% * last_resid : the last residue
% 
% Example:
% >> load superpos
% >> plot(sig)                               
% >> [atoms, Rl]=mpdecomp(sig, 300, 8);
% >> figure;plot(Rl); %% Residue   
% >> figure;plot(sig-Rl); %% Aproximation

if ~exist('phi')
  phi = 1;
end

phitphi = phi'*phi;

nsig = length(sig);

%% Atoms encodes all requi
atoms = zeros(nbiter, 5);

log2_sc_max = log2(nsig) - 1;
sc = 2.^(linspace(0, log2_sc_max, nbscales));

w = (0:2/nsig:(2-1/nsig));
w = pi*(w - 2*(w>(1-1/nsig)));

%% Initialization
Rm = sig * phitphi'; %% Residue
rsig = Rm;

for m = 1:nbiter,
  tRm = fft(Rm);
  last_resid_norm = norm(Rm(:));
  
  best_scp = 0;
  
  best_pos = [];
  best_sc = [];
  best_sc_ind = [];
  
  for s = 1:nbscales,
    twav = sc(s)^.5 * mex1d_freq(sc(s)*w);
    scp = ifft(tRm .* twav);
    [m_scp, m_pos] = max(abs(scp));
    
    if (abs(best_scp) < m_scp)
      best_scp = scp(m_pos);
      best_pos = m_pos;
      best_sc = sc(s);
      best_sc_ind = s;
    end
  end 
  
  Rm = Rm - best_scp .* mex1d(((1:nsig) - best_pos)/best_sc)/best_sc^.5; 
  atoms(m,:) = [best_sc_ind best_sc best_pos best_scp last_resid_norm];
  fprintf('m=%i: SNR=%f\n', m, mysnr(sig, rsig - Rm));
end

last_resid = Rm;
fprintf('\n');

function snr = mysnr(psig, nsig)
snr = 20*log10(std2(psig) / std2(psig - nsig));