function [atoms, last_resid] = mpdecomp_ga(sig, nbiter, nbscales, ga_iter, nb_batom)
% Manifold Gradient Ascent optimized Matching pursuit decomposition 
% of 1-D signal using discrete dico based on Mexican Hat
%
% See also : mex1d.m and mex1d_freq.m

% INPUTS: 
% * sig : original signal
% * nbiter : number of iterations
% * nbscales : number of scales between 1 length(sig)/2
% * ga_iter : number of iterations inside Gradient Ascent optimization
%             (if set to 0, no optimization, i.e. this is the usual MP) 
% * nb_batom : number of best atoms to keep at each iteration.
%
% OUTPUTS:
% * atoms : a nbiter x 3 matrix where column 1 gives the optimized
% position, column 2 the optimized scale, and column 3 the
% optimized scalar product at each MP iteration.
% * last_resid : the last residue
% 
% Example: (see bench.m for a more detailed example)
% >> load superpos
% >> plot(sig)                       
% %% No optimization
% >> [atoms, Rl]=mpdecomp(sig, 300, 8);
% >> figure;plot(Rl); %% Residue   
% >> figure;plot(sig-Rl); %% Aproximation
% %% Optimization
% >> [atoms, Rl]=mpdecomp_ga(sig, 300, 8, 100);
% >> figure;plot(Rl); %% Residue   
% >> figure;plot(sig-Rl); %% Aproximation
% 
%
% REQUIRES:
% * makecol

nsig = length(sig);
sig = makecol(sig);
t = makecol(1:nsig);

%% Atoms encodes all requi
atoms = zeros(nbiter, 3);

log2_sc_max = log2(nsig) - 1;
sc = 2.^(linspace(0, log2_sc_max, nbscales));

w = makecol(0:2/nsig:(2-1/nsig));
w = pi*(w - 2*(w>(1-1/nsig)));

%% Initialization
Rm = sig; %% Residue

for m = 1:nbiter,
  tRm = fft(Rm);
  
  best_scp = 0;
  
  best_pos = [];
  best_sc = [];
  best_sc_ind = [];
  
  for s = 1:nbscales,
    twav = sc(s)^.5 * mex1d_freq(sc(s)*w);
    scp = ifft(tRm .* twav);
    [m_scp, m_pos] = max(abs(scp));
    
    if (abs(best_scp) < m_scp)
      best_scp = scp(m_pos);
      best_pos = m_pos;
      best_sc = sc(s);
      best_sc_ind = s;
    end
  end 
  
  %% Optimization of the best atom using Gradient Ascent
  [lambda_opt, atom_opt] = ga_atom(Rm, t, @mex1d_lambda, ...
				   [best_pos best_sc], ga_iter);
  scp_opt = Rm' * atom_opt;
  
  %% Residue computation
  Rm = Rm - scp_opt * atom_opt;
  
  atoms(m,:) = [makerow(lambda_opt) scp_opt];
  
  fprintf('m=%i: SNR=%f\n', m, mysnr(sig, sig - Rm));
end

last_resid = Rm.';
fprintf('\n');

function snr = mysnr(psig, sig)
snr = 20*log10(std2(psig) / std2(psig - sig));

function Kfirst = keepKfirst(v,K)

[ans, pos] = sort(abs(v(:)));
Kfirst = v(pos(1:min(end,K)));

