function [p, atom] = ga_atom_2d(img, X, Y, atomfct, p0, p_low, p_high, nb_iter, p_up)
% Manifold Gradient Ascent convergence for any real atom of a dictionary
%
% INPUTS:
%  img : orginal image to fit
%  X,Y : the positions matrices (such as provided by meshgrid())
%  atomfct : function handle linked to the atom function. atomfct MUST
%  take as parameters (X,Y, p0) in this order ! (see for instance
%  gen_mh.m)
%  p0 : initial parameter for the convergence
%  nb_iter : number of iteration for the convergence
%  p_low : lower bounds on parameters
%  p_high : upper bounds on parameters
%  p_up : position of the parameter to update (default: 1:length(p0))
%
% OUTPUTS:
%  p : new parameter obtained from NG
%  atom : corresponding atom
%
% REQUIRES:
%  makecol 

[nrow, ncol] = size(img);
npix = nrow*ncol;
line_search_nbiter = 10;

img = makecol(img(:));

p = makecol(p0);

if (~exist('p_up'))
  nb_param = length(p0);
  p_up = 1:nb_param;
else
  nb_param = length(p_up);
end

atom = atomfct(X, Y, p);
atom = makecol(atom(:));

img_dot_atom = atom'*img;

%fprintf('|<img|atom>|: %f\n', abs(img_dot_atom));

epsilon = 0.00001;
alpha = 0.1;

for iter = 1:nb_iter,

  %% Computing the gradient and the Hessian
  %% using a bunch of matrix computation
  %% For full information, see the paper 
  %% (Eqt (22)-(30))
  J = zeros(nb_param, 1);
  datom = zeros(npix, nb_param);
  
  for k = 1:nb_param,
    np = p;
    np(p_up(k)) = np(p_up(k)) + epsilon;
    natom = atomfct(X, Y, np);
    natom = makecol(natom(:));
    datom(:,k) = (natom - atom)/epsilon;
  end
  
  %% The Gradient (forget the positive constant since normalization
  %% after by the norm of J)
  J = img_dot_atom * (datom)'*img;
  
  %% The metric
  G = datom'*datom;
  
  %% Inverse of the Metric is computed by pseudoinverse for
  %% stability
  try
    Gpinv = pinv(G);
  catch
    break;
  end
  
  normJ = (J'*Gpinv*J)^.5;
  
  %% Optimization step
  dp_up = (Gpinv*J/normJ)*alpha;
  good_update = 0;
  
  line_search_iter = 0;
  while (good_update == 0)
    line_search_iter = line_search_iter + 1;
    
    np(p_up) = bound_param(p(p_up) + dp_up, p_low(p_up), p_high(p_up));
    
    natom = atomfct(X, Y, np);
    natom = makecol(natom(:));
    new_img_dot_atom = natom'*img;
    good_update = abs(new_img_dot_atom) > abs(img_dot_atom);
    
    dp_up = dp_up / 2;
    
    if ((line_search_iter > line_search_nbiter) | (max(abs(dp_up)) < eps))
      break;
    end
  end
  
  %% checking if it is worthy to continue
  if (good_update)
    p = np;
    img_dot_atom = new_img_dot_atom;
    atom = natom;
    %fprintf('|<img|atom>|: %f\n', abs(img_dot_atom));
  else
    %disp('Nothing to search anymore');
    iter = iter - 1;
    break;
  end  
  %end
end

%fprintf(' (%i/%i) -> ', iter, nb_iter);
atom = reshape(atom, [nrow ncol]);


function np = bound_param(p, p_low, p_high)

np = max( min( p, makecol(p_high)), ...
	  makecol(p_low) );

    