% Monte Carlo Simulation on the reconstruction of K-spars signals from
% their quantized compressive observations (for Gaussian sensing).
%
% This script generates Fig1(left) in the paper: 
% A. Moshtaghpour, L. Jacques, V. Cambareri, K. Degraux, 
% and C. De Vleesschouwer, "Consistent Basis Pursuit for Signal and Matrix Estimates in Quantized Compressed Sensing"

% Initializing
clc;clear all;
addpath('.\mod_unlocbox');

% setting parameters for solver
param_solver.verbose = 1;          % display parameter
param_solver.maxit = 60000;        % maximum number of iterations
param_solver.tol = 1e-7;           % tolerance to stop iterating
param_solver.gamma = 5e-3;         % proximal operator parameter
param_solver.lambda = 0.99;

% Loading toolbox
init_unlocbox();

% Size of the signal
N = 2048;

% Sparsity level
K = 16;

% Number of bits
B = 3;

% Number of Monte-Carlo iterations
mont = 5;

% Number of measurements
j = 0.5*(0:6);
M = round(K*2.^(j+4));

MSE_mont_cobp = [];average_MSE_cobp = []; MSE_mont_bpdn = [];average_MSE_bpdn = [];
for i1=1:length(M)
    rng(77);
    fprintf('M/K = %g \n', M(i1)/K);
    
    for i2=1:mont
        fprintf('Iteration = %g / %g \n', i2, mont);
        
        %  Generating mesurements matrix
        A = randn(M(i1), N);
        
        %  Generating signal
        idx = randperm(N);
        x = zeros(N,1);
        x(idx(1:K)) = randn(K,1);
        x = x/norm(x);
        
        %  consistency radius
        delta = 6/2^(B-1);
        epsilon = func_lp_ball_epsilon(2,delta,M(i1));
        
        %  dithering
        dithering = -delta/2+delta*rand(M(i1),1);
        
        %  Measurements
        y = A * x + dithering;
        
        %  Quantizing the measurements
        yq = func_uniform_quantize(y,delta);
        
        %  solving problems
        param_solver.init_pt = zeros(N,1);
        xhat_bpdn = func_BPDN(yq-dithering,A,epsilon,param_solver);

        % not used in the paper: a warm start for CoBP starting from BPDN solution 
        param_solver.init_pt = xhat_bpdn;
        xhat_cobp = func_CoBP(yq-dithering,A,delta,param_solver);
        
        %  output snr for each monte-carlo trail
        mse_mont_cobp(i2,i1) = norm(x-xhat_cobp);
        mse_mont_bpdn(i2,i1) = norm(x-xhat_bpdn);
        
    end
    
    % Averaging MSEs of Monte-Carlo iterations
    average_MSE_cobp(i1) = log2(mean(mse_mont_cobp(:,i1)));
    average_MSE_bpdn(i1) = log2(mean(mse_mont_bpdn(:,i1)));
    
    % Applying linear fit
    p_cobp= polyfit(log2(M(1:i1)/K),average_MSE_cobp(1:i1),1);
    fit_cobp  = polyval(p_cobp,log2(M(1:i1)/K));
    
    p_bpdn= polyfit(log2(M(1:i1)/K),average_MSE_bpdn(1:i1),1);
    fit_bpdn  = polyval(p_bpdn,log2(M(1:i1)/K));
    
    % Plotting
    close all
    figure
    plot(log2(M(1:i1)/K),average_MSE_cobp,'k-o','linewidth',2); hold on
    plot(log2(M(1:i1)/K),average_MSE_bpdn,'r-s','linewidth',2);hold on
    plot(log2(M(1:i1)/K),fit_cobp,'k--','linewidth',1);hold on
    plot(log2(M(1:i1)/K),fit_bpdn,'r--','linewidth',1);
    xlabel('log_2(M/K)')
    ylabel('log_2(||x-x_o||)')
    title(sprintf('Fig1(left): Gaussian + %i sparse signal, B = %i ',K,B))
    legend('CoBP','BPDN',['CoBP linear fit, slope = ',num2str(p_cobp(1))],['BPDN linear fit, slope = ',num2str(p_bpdn(1))])
    
end
% Closing the toolbox
close_unlocbox();

