% Monte Carlo Simulation on the reconstruction of rank-1 matrices from
% their quantized compressive observations (for Gaussian sensing).
%
% This script generates Fig1(right) in the paper: 
% A. Moshtaghpour, L. Jacques, V. Cambareri, K. Degraux, 
% and C. De Vleesschouwer, "Consistent Basis Pursuit for Signal and Matrix Estimates in Quantized Compressed Sensing"

% Initializing
clc;clear all;
addpath('.\mod_unlocbox');

% setting parameters for solver
param_solver.verbose = 1;          % display parameter
param_solver.maxit = 60000;        % maximum number of iterations
param_solver.tol = 1e-6;           % tolerance to stop iterating
param_solver.gamma = 5e-3;         % proximal operator parameter
param_solver.lambda = 0.99;

% Loading toolbox
init_unlocbox();

% Size of the signal
N = 1024;
n = sqrt(N);

% rank of the sensed matrix
r = 1;
p = 64;

% Number of bins
B = 2;

% Number of monte-carlo trails (20 for the paper)
mont = 5;

% Number of measurements
j = 2+0.5*(0:6);
M = round(p*2.^j);

MSE_cobp = []; MSE_bpdn = [];
for i1=1:length(M)
    fprintf('M/p = %g \n', 2^j(i1));
    
    E_mont_cobp = [];E_mont_bpdn = [];
    
    for i2=1:mont
        fprintf('Iteration = %g \n', i2);
        
        % Generating the mesurements matrix
        A = randn(M(i1), N);
        
        % Generating a rank r matrix
        x0 = randn(n,r);
        X = x0*x0';
        X = X/norm(X,'fro');
        x = X(:);
        
        % Consistency radius
        delta = 6/2^(B-1);
        epsilon=func_lp_ball_epsilon(2,delta,M(i1));
        
        % dithering
        dithering = -delta/2+delta*rand(M(i1),1);
        
        
        % Measurements
        y = A * x + dithering;
        
        % Quantizing the measurements
        yq = func_uniform_quantize(y,delta);
        
        % Solving the problem
        param_solver.init_pt = zeros(N,1);
        xhat_bpdn = func_BPDN_matrix_completion(yq-dithering,A,epsilon,param_solver);
        
        % not used in the paper: a warm start for CoBP starting from BPDN solution 
        param_solver.init_pt = xhat_bpdn;
        xhat_cobp = func_CoBP_matrix_completion(yq-dithering,A,delta,param_solver);
        
        % output mse for each monte-carlo trail
        E_mont_cobp(i2) = norm(x-xhat_cobp);
        E_mont_bpdn(i2) = norm(x-xhat_bpdn);
    end
    
    % output snr for each oversampling value
    MSE_cobp(i1) = log2(mean(E_mont_cobp));
    MSE_bpdn(i1) = log2(mean(E_mont_bpdn));
    
    % Applying linear fit
    p_cobp= polyfit(log2(M(1:i1)/p),MSE_cobp(1:i1),1);
    fit_cobp  = polyval(p_cobp,log2(M(1:i1)/p));
    
    p_bpdn= polyfit(log2(M(1:i1)/p),MSE_bpdn(1:i1),1);
    fit_bpdn  = polyval(p_bpdn,log2(M(1:i1)/p));
    
    % Plotting
    close all
    figure
    plot(log2(M(1:i1)/p),MSE_cobp(1:i1),'k-o','linewidth',2); hold on
    plot(log2(M(1:i1)/p),MSE_bpdn(1:i1),'r-s','linewidth',2); hold on
    plot(log2(M(1:i1)/p),fit_cobp,'k--','linewidth',1);hold on
    plot(log2(M(1:i1)/p),fit_bpdn,'r--','linewidth',1);
    
    xlabel('log_2(M/P)')
    ylabel('log_2(||x-x_o||)')
    legend('CoBP','BPDN',['CoBP linear fit, slope = ',num2str(p_cobp(1))],['BPDN linear fit, slope = ',num2str(p_bpdn(1))])
    grid on
    title(sprintf('Fig1(right): Gaussian + rank %i sensed signal, B = %i ',r,B))
    
    
end
% Closing the toolbox
close_unlocbox();
