function [sol, info] = proj_binf(x, ~, param)

% Start the time counter
t1 = tic;

% Optional input arguments
if ~isfield(param, 'y'), param.y = 0; end
if ~isfield(param, 'A'), param.A = @(x) x; param.At = @(x) x; end
if ~isfield(param, 'epsilon'), param.epsilon = 1e-3; end
if ~isfield(param, 'tight'), param.tight = 1; end
if ~isfield(param, 'tol'), param.tol = 1e-3; end
if ~isfield(param, 'verbose'), param.verbose = 1; end
if ~isfield(param, 'nu'), param.nu = 1; end
if ~isfield(param, 'maxit'), param.maxit = 200; end


% Useful functions for the projection
sc = @(z) z.*min(param.epsilon./abs(z(:)), 1); % scaling

% Projection
if param.tight % TIGHT FRAME CASE
    
    temp = param.A(x) - param.y;
    sol = x + 1/param.nu * param.At(sc(temp)-temp);
    crit = 'TOL_EPS'; iter = 0; u = NaN;
    
else % NON TIGHT FRAME CASE
    
    % Initializations
    sol = x; u = zeros(size(param.y));
    iter = 0; doloop = 1;
    
    % Tolerance onto the ball
    epsilon_low = param.epsilon/(1+param.tol);
    epsilon_up = param.epsilon/(1-param.tol);
    
    % Check if we are in the ball
    norm_res = norm(param.y-param.A(sol), 'inf');
    if norm_res <= epsilon_up
        crit = 'IN_BALL'; doloop = 0;
    end
    
    % Projection onto the ball
    % Init
    while doloop
        % Update number of iteration
        iter = iter + 1;
        
        % Residual
        res = param.A(sol) - param.y; norm_res = norm(res(:), inf);
        
        % Scaling for the projection
        res = u*param.nu + res; norm_proj = abs(res(:));
        
        
        % Stopping criterion
        if (norm_res>=epsilon_low && norm_res<=epsilon_up)
            crit = 'TOL_EPS'; break;
        elseif iter >= param.maxit
            crit = 'MAX_IT'; break;
        end
        
        
        ratio = min(1, param.epsilon./norm_proj);
        
        
        u = 1/param.nu * (res - res.*ratio);
        
        % Current estimate
        sol = x - param.At(u);
        
        
    end
end

% Log after the projection onto the ball
if param.verbose >= 1
    temp = param.A(sol);
    fprintf(['  Proj. Binf: epsilon = %e, ||y-Ax||_inf = %e,', ...
        ' %s, iter = %i\n'], param.epsilon, norm(param.y(:)-temp(:),inf), ...
        crit, iter);
end

% Infos about algorithm
info.algo=mfilename;
info.iter=iter;
info.final_eval=norm(param.A(sol) - param.y);
info.crit=crit;
info.residue=u;
info.time=toc(t1);

end

