% Monte Carlo Simulation on the discrepency between Bernoulli and 
% Gaussian quantized compressed sensing of sparse signals when K increases.
%
% This script generates Fig1(middle) in the paper: 
% A. Moshtaghpour, L. Jacques, V. Cambareri, K. Degraux, 
% and C. De Vleesschouwer, "Consistent Basis Pursuit for Signal and Matrix Estimates in Quantized Compressed Sensing"


% Initializing
clc;clear all;
addpath('mod_unlocbox');

% setting parameters for solver
param_solver.verbose = 1;          % display parameter
param_solver.maxit = 60000;         % maximum number of iterations
param_solver.tol = 1e-8;          % tolerance to stop iterating
param_solver.gamma = 5e-3;         % stepsize
param_solver.lambda = 0.99;

% Loading toolbox
init_unlocbox();

% Size of the signal
N = 1024;

% Sparsity vector
K = 2.^(0:6);

% Number of bins
B = 4;

% Number of monte-carlo trails (20 for the paper)
mont = 5; 

% Run for different oversampling values (I know this type of initializing increases elapsed time ;-))
MSE_bernoulli = [];MSE_bernoulli_lambda = [];MSE_gaussian = [];


for i1=1:length(K)
    fprintf('K = %g \n', K(i1));
    
    % Number of measurements
    M = 16*K(i1);
    
    % Reseting mse vectors of mont-carlo iterations (see line 27 ;-))
    mse_mont_bernoulli = [];mse_mont_bernoulli_lambda = [];mse_mont_gaussian = [];
    
    for i2=1:mont
        fprintf('Iteration = %g / %g \n', i2, mont);
        
        % Generating mesurements matrix
        A_bernoulli = 2*(rand(M,N)<0.5)-1;
        A_gaussian = randn(M,N);
        
        
        % Generating a K sparse signal
        x = zeros(N,1);
        idx = randperm(N)';
        x(idx(1:K(i1))) = randn(K(i1),1);
        x = x/norm(x);

        % Specifying lambda for CoBP lambda
        lambda = norm(x,inf);
        
        % Measurements
        y_bernoulli = A_bernoulli * x;
        y_gaussian = A_gaussian * x;
        
        % Width of the quantizer bins
        delta = 6/2^(B-1);
        
        % Generating uniform dithering (note: randc in [-0.5,0.5])
        % Note: theory saus dithering in [0, delta] but [-delta/2, delta/2]
        % is equivalent.
        dithering = delta*randc(M,1);
        
        % Adding dithering to the measurements
        y_bernoulli = y_bernoulli + dithering;
        y_gaussian = y_gaussian + dithering;
        
        % Quantizing the measurements
        yq_bernoulli = func_uniform_quantize(y_bernoulli,delta);
        yq_gaussian = func_uniform_quantize(y_gaussian,delta);
        
        % Solving the problems
        
        % CoBP Bernoulli
        param_solver.init_pt = zeros(N,1);
        xhat_bernoulli = func_CoBP(yq_bernoulli-dithering,A_bernoulli,delta,param_solver);

        % CoBP_lambda Bernoulli ... plus a warm start for stabilizing it
        param_solver.init_pt = xhat_bernoulli;         
        xhat_bernoulli_lambda = func_CoBP_lambda(yq_bernoulli-dithering,A_bernoulli,delta,lambda,param_solver);
        
        % CoBP Gaussian
        param_solver.init_pt = zeros(N,1);
        xhat_gaussian = func_CoBP(yq_gaussian-dithering,A_gaussian,delta,param_solver);
        
        % MSE for each monte-carlo trail
        mse_mont_bernoulli_lambda(i2) = norm(x-xhat_bernoulli_lambda);
        mse_mont_bernoulli(i2) = norm(x-xhat_bernoulli);
        mse_mont_gaussian(i2) = norm(x-xhat_gaussian);
    end
    
    %  MSE for each point on horizontal axis
    MSE_bernoulli_lambda(i1) = log2(mean(mse_mont_bernoulli_lambda));
    MSE_bernoulli(i1) = log2(mean(mse_mont_bernoulli));
    MSE_gaussian(i1) = log2(mean(mse_mont_gaussian));
    
    % Plotting the results
    close all
    figure
    plot(log2(K(1:i1)),MSE_bernoulli_lambda,'k-o','linewidth',2); hold on
    plot(log2(K(1:i1)),MSE_bernoulli,'r-s','linewidth',2);hold on
    plot(log2(K(1:i1)),MSE_gaussian,'b-','linewidth',2);
    xlabel('log_2(K)')
    ylabel('log_2(||x-x_o||)')
    title(sprintf('Fig1(middle): B = %i, M/K = %i',B, 16))
    legend('CoBP_{\lambda} + Bernoulli','CoBP + Bernoulli','CoBP + Gaussian');
end

%% Closing the toolbox
close_unlocbox();

